<?php
require_once 'config.php';
require_once 'lib/ACMECert.php';

use skoerfgen\ACMECert\ACMECert;

session_start();

// --- LANGUAGE LOGIC ---
if (isset($_GET['lang'])) {
    $_SESSION['lang'] = $_GET['lang'] === 'en' ? 'en' : 'es';
}
$lang = $_SESSION['lang'] ?? 'es';

$t = [
    'es' => [
        'title' => 'SSL FREE - Generador ACME v3',
        'subtitle' => 'Genera certificados Let\'s Encrypt mediante validación HTTP-01 automáticamente.',
        'login_title' => 'Iniciar Sesión',
        'login_subtitle' => 'Herramienta SSL ACME v3',
        'username' => 'Usuario',
        'password' => 'Contraseña',
        'login_btn' => 'Entrar',
        'forgot_pass' => '¿Olvidaste tu contraseña?',
        'recovery_btn' => 'Enviar Correo de Recuperación',
        'back_to_login' => 'Volver al inicio',
        'recovery_notice' => 'Se enviará un correo a la dirección de recuperación configurada para restablecer tu contraseña.',
        'setup_title' => 'Configuración Inicial',
        'setup_subtitle' => 'Por seguridad, debes actualizar tus credenciales de acceso.',
        'setup_email_label' => 'Tu Correo Electrónico (será tu nuevo usuario)',
        'setup_pass_label' => 'Nueva Contraseña',
        'setup_notice' => 'Este correo se usará también para la recuperación de contraseña en caso de olvido.',
        'setup_btn' => 'Guardar y Continuar',
        'domains_label' => 'Dominios (separados por coma)',
        'add_subdomains' => '+ Añadir Subdominios cPanel',
        'generate_btn' => 'Generar / Renovar SSL',
        'copy_crt' => 'Copiar CRT',
        'copy_key' => 'Copiar KEY',
        'copy_bundle' => 'Copiar CABUNDLE',
        'user_settings' => 'Configuración de Usuario',
        'new_user' => 'Nuevo Usuario (opcional)',
        'new_pass' => 'Nueva Contraseña (opcional)',
        'recovery_email' => 'Correo de Recuperación',
        'tracking_consent' => 'Doy mi consentimiento para el seguimiento de mejoras publicadas y anuncios relevantes.',
        'update_btn' => 'Actualizar Configuración',
        'logout' => 'Cerrar Sesión',
        'admin_login' => 'Iniciar Sesión (Admin)',
        'created_by' => 'Creado por',
        'instructions_title' => 'Instrucciones de Uso',
        'instr_1' => '1. Ingresa el dominio principal (ej: tudominio.com).',
        'instr_2' => '2. Usa "+ Añadir Subdominios cPanel" si necesitas www, mail, etc.',
        'instr_3' => '3. Haz clic en "Generar / Renovar SSL".',
        'instr_4' => '4. Copia los códigos resultantes en tu panel de control (cPanel, Plesk, etc).',
        'copy_success' => 'Copiado al portapapeles',
        'staging' => 'Staging',
        'production' => 'Producción',
        'placeholder_domain' => 'ej: canva.com, www.canva.com',
        'settings_updated' => 'Configuración actualizada correctamente.',
        'setup_complete' => 'Configuración completada. ¡Bienvenido!',
        'setup_error' => 'Por favor ingresa un correo válido y una contraseña.',
        'credentials_error' => 'Credenciales incorrectas o código expirado.',
        'recovery_sent' => 'Se ha enviado un código de recuperación a: ',
        'recovery_error' => 'Error al enviar el correo. Contacta al soporte.',
        'no_recovery_email' => 'No hay un correo de recuperación configurado.',
        'ssl_success_cpanel' => 'SSL Generado e Instalado exitosamente en cPanel!',
        'ssl_success_manual' => 'SSL Generado exitosamente. Por favor instalar manualmente.',
        'like_tool' => '¿Te gusta esta herramienta?',
        'lang_toggle' => 'English'
    ],
    'en' => [
        'title' => 'SSL FREE - ACME v3 Generator',
        'subtitle' => 'Generate Let\'s Encrypt certificates via HTTP-01 validation automatically.',
        'login_title' => 'Login',
        'login_subtitle' => 'SSL Tool ACME v3',
        'username' => 'Username',
        'password' => 'Password',
        'login_btn' => 'Login',
        'forgot_pass' => 'Forgot your password?',
        'recovery_btn' => 'Send Recovery Email',
        'back_to_login' => 'Back to Home',
        'recovery_notice' => 'An email will be sent to the configured recovery address to reset your password.',
        'setup_title' => 'Initial Setup',
        'setup_subtitle' => 'For security, you must update your access credentials.',
        'setup_email_label' => 'Your Email Address (will be your new username)',
        'setup_pass_label' => 'New Password',
        'setup_notice' => 'This email will also be used for password recovery in case it is forgotten.',
        'setup_btn' => 'Save and Continue',
        'domains_label' => 'Domains (comma separated)',
        'add_subdomains' => '+ Add cPanel Subdomains',
        'generate_btn' => 'Generate / Renew SSL',
        'copy_crt' => 'Copy CRT',
        'copy_key' => 'Copy KEY',
        'copy_bundle' => 'Copy CABUNDLE',
        'user_settings' => 'User Settings',
        'new_user' => 'New Username (optional)',
        'new_pass' => 'New Password (optional)',
        'recovery_email' => 'Recovery Email',
        'tracking_consent' => 'I consent to tracking published improvements and relevant announcements.',
        'update_btn' => 'Update Settings',
        'logout' => 'Logout',
        'admin_login' => 'Admin Login',
        'created_by' => 'Created by',
        'instructions_title' => 'Usage Instructions',
        'instr_1' => '1. Enter the primary domain (e.g., yourdomain.com).',
        'instr_2' => '2. Use "+ Add cPanel Subdomains" if you need www, mail, etc.',
        'instr_3' => '3. Click on "Generate / Renew SSL".',
        'instr_4' => '4. Copy the resulting codes into your control panel (cPanel, Plesk, etc).',
        'copy_success' => 'Copied to clipboard',
        'staging' => 'Staging',
        'production' => 'Production',
        'placeholder_domain' => 'e.g., canva.com, www.canva.com',
        'settings_updated' => 'Settings updated successfully.',
        'setup_complete' => 'Setup completed. Welcome!',
        'setup_error' => 'Please enter a valid email and password.',
        'credentials_error' => 'Incorrect credentials or code expired.',
        'recovery_sent' => 'A recovery code has been sent to: ',
        'recovery_error' => 'Error sending email. Contact support.',
        'no_recovery_email' => 'No recovery email address configured.',
        'ssl_success_cpanel' => 'SSL Generated and Successfully Installed in cPanel!',
        'ssl_success_manual' => 'SSL Generated successfully. Please install manually.',
        'like_tool' => 'Support this tool?',
        'lang_toggle' => 'Español'
    ]
];

$tr = $t[$lang];

// --- SETTINGS HELPERS ---
function load_settings()
{
    $defaults = [
        'username' => APP_USERNAME,
        'password_hash' => APP_PASSWORD_HASH,
        'recovery_email' => '',
        'tracking_consent' => false,
        'recovery_code' => '',
        'recovery_expiry' => 0,
        'needs_password_change' => false
    ];
    if (!file_exists(SETTINGS_FILE)) {
        return $defaults;
    }
    $data = json_decode(file_get_contents(SETTINGS_FILE), true);
    return array_merge($defaults, $data ?: []);
}

function save_settings($settings)
{
    if (!is_dir(DATA_DIR))
        mkdir(DATA_DIR, 0755, true);
    file_put_contents(SETTINGS_FILE, json_encode($settings, JSON_PRETTY_PRINT));
}

$app_settings = load_settings();

// --- AUTH LOGIC ---
if (isset($_POST['login'])) {
    $user = $_POST['username'] ?? '';
    $pass = $_POST['password'] ?? '';

    // Check normal login or recovery code login
    if (
        ($user === $app_settings['username'] && password_verify($pass, $app_settings['password_hash'])) ||
        ($user === $app_settings['username'] && !empty($app_settings['recovery_code']) && $pass === $app_settings['recovery_code'] && time() < $app_settings['recovery_expiry'])
    ) {

        $_SESSION['authenticated'] = true;

        // If they used a recovery code, they MUST change password
        if ($pass === $app_settings['recovery_code']) {
            $app_settings['needs_password_change'] = true;
            $app_settings['recovery_code'] = ''; // Clear code
            save_settings($app_settings);
        }
    } else {
        $error = $tr['credentials_error'];
    }
}

if (isset($_GET['logout'])) {
    session_destroy();
    header("Location: index.php");
    exit;
}

if (!isset($_SESSION['authenticated']) && !ENABLE_PUBLIC_ACCESS) {
    // --- RECOVERY LOGIC (6-DIGIT CODE) ---
    if (isset($_POST['recover_password'])) {
        $recovery_email = $app_settings['recovery_email'];
        if ($recovery_email) {
            $code = str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);
            $app_settings['recovery_code'] = $code;
            $app_settings['recovery_expiry'] = time() + (15 * 60); // 15 minutes
            save_settings($app_settings);

            $to = $recovery_email;
            $subject = "Recovery Code - SSL FREE";
            if ($lang === 'es') {
                $message = "Has solicitado recuperar tu acceso.\n\n";
                $message .= "Tu código de acceso temporal es: " . $code . "\n";
                $message .= "Este código expirará en 15 minutos.\n\n";
                $message .= "Usa este código como contraseña en la pantalla de inicio de sesión.\n";
            } else {
                $message = "You have requested to recover your access.\n\n";
                $message .= "Your temporary access code is: " . $code . "\n";
                $message .= "This code will expire in 15 minutes.\n\n";
                $message .= "Use this code as your password on the login screen.\n";
            }

            $headers = "From: noreply@" . $_SERVER['HTTP_HOST'] . "\r\n";
            if (@mail($to, $subject, $message, $headers)) {
                $status_message = $tr['recovery_sent'] . $recovery_email;
            } else {
                $status_error = $tr['recovery_error'];
            }
        } else {
            $status_error = $tr['no_recovery_email'];
        }
    }

    render_login($error ?? null, $status_message ?? '', $status_error ?? '');
    exit;
}

// --- MANDATORY SETUP CHECK ---
if ($app_settings['username'] === 'admin' || $app_settings['needs_password_change']) {
    if (isset($_POST['mandatory_setup'])) {
        $new_email = $_POST['setup_email'];
        $new_pass = $_POST['setup_password'];

        if (filter_var($new_email, FILTER_VALIDATE_EMAIL) && !empty($new_pass)) {
            $app_settings['username'] = $new_email;
            $app_settings['recovery_email'] = $new_email;
            $app_settings['password_hash'] = password_hash($new_pass, PASSWORD_BCRYPT);
            $app_settings['needs_password_change'] = false;
            save_settings($app_settings);
            $status_message = $tr['setup_complete'];
        } else {
            $setup_error = $tr['setup_error'];
        }
    }

    if ($app_settings['username'] === 'admin' || $app_settings['needs_password_change']) {
        render_mandatory_setup($setup_error ?? null);
        exit;
    }
}

// --- APP LOGIC ---
$status_message = "";
$certs = ['crt' => '', 'key' => '', 'bundle' => ''];

if (isset($_POST['generate_ssl'])) {
    try {
        $domain_input = $_POST['domain'] ?: (FORCE_DOMAIN ?: $_SERVER['HTTP_HOST']);
        $domains = array_map('trim', explode(',', $domain_input));
        $domain = $domains[0]; // Primary domain for display/logs
        $email = LE_EMAIL;
        $staging = !LE_PRODUCTION;

        $acme = new ACMECert(!$staging);

        // Load or Create Account Key
        $account_key_file = DATA_DIR . '/account.key';
        if (!file_exists($account_key_file)) {
            $account_key = $acme->generateRSAKey(4096);
            file_put_contents($account_key_file, $account_key);
            chmod($account_key_file, 0600);
        } else {
            $account_key = file_get_contents($account_key_file);
        }

        $acme->loadAccountKey($account_key);
        $acme->register(true, [$email]);

        // Generate Domain Key
        $domain_key = $acme->generateRSAKey(2048);

        // Challenge Handler
        $challenge_cb = function ($opts) use ($lang) {
            $path = $_SERVER['DOCUMENT_ROOT'] . $opts['key'];
            $dir = dirname($path);
            if (!is_dir($dir))
                mkdir($dir, 0755, true);
            file_put_contents($path, $opts['value']);
            return function () use ($path) {
                @unlink($path);
            };
        };

        // Get Certificate
        $chain = $acme->getCertificateChain($domain_key, $domains, $challenge_cb);

        // Split Chain (Leaf vs Bundle)
        $delim = '-----END CERTIFICATE-----';
        $parts = explode($delim, trim($chain));
        $certs['crt'] = trim($parts[0] . $delim);
        $certs['key'] = $domain_key;
        $certs['bundle'] = trim(implode($delim, array_slice($parts, 1)));

        // Try cPanel Auto-Install
        if (CPANEL_API_TOKEN && CPANEL_USER) {
            install_to_cpanel($domain, $certs['crt'], $certs['key'], $certs['bundle']);
            $status_message = $tr['ssl_success_cpanel'];
        } else {
            $status_message = $tr['ssl_success_manual'];
        }

        // --- NOTIFICATION ---
        $to = NOTIFICATION_EMAIL;
        $subject = "Nuevo SSL Generado: $domain";
        $message = "Se ha generado un nuevo certificado SSL.\n\n";
        $message .= "Dominio: $domain\n";
        $message .= "Email ACME: $email\n";
        $message .= "Fecha: " . date('Y-m-d H:i:s') . "\n";
        $message .= "Consentimiento de seguimiento: " . ($app_settings['tracking_consent'] ? 'SÍ' : 'NO') . "\n";
        $message .= "Correo de recuperación: " . ($app_settings['recovery_email'] ?: 'No definido') . "\n";

        $headers = "From: noreply@" . $_SERVER['HTTP_HOST'] . "\r\n";
        @mail($to, $subject, $message, $headers);

    } catch (Exception $e) {
        $status_error = "Error: " . $e->getMessage();
    }
}

// --- UPDATE SETTINGS LOGIC ---
if (isset($_POST['update_settings'])) {
    $new_user = $_POST['new_username'] ?: $app_settings['username'];
    $new_pass = $_POST['new_password'];
    $recovery_email = $_POST['recovery_email'];
    $consent = isset($_POST['tracking_consent']);

    $app_settings['username'] = $new_user;
    if (!empty($new_pass)) {
        $app_settings['password_hash'] = password_hash($new_pass, PASSWORD_BCRYPT);
    }
    $app_settings['recovery_email'] = $recovery_email;
    $app_settings['tracking_consent'] = $consent;

    save_settings($app_settings);
    $status_message = $tr['settings_updated'];

    // --- NOTIFICATION OF SETTINGS UPDATE ---
    $to = NOTIFICATION_EMAIL;
    $subject = "Ajustes Actualizados en: " . $_SERVER['HTTP_HOST'];
    $message = "Se han actualizado los ajustes del Generador SSL.\n\n";
    $message .= "Dominio: " . $_SERVER['HTTP_HOST'] . "\n";
    $message .= "Nuevo Usuario: " . $app_settings['username'] . "\n";
    $message .= "Correo de Recuperación: " . ($app_settings['recovery_email'] ?: 'No definido') . "\n";
    $message .= "Consentimiento Seguimiento: " . ($app_settings['tracking_consent'] ? 'SÍ' : 'NO') . "\n";
    $message .= "Fecha: " . date('Y-m-d H:i:s') . "\n";

    $headers = "From: noreply@" . $_SERVER['HTTP_HOST'] . "\r\n";
    @mail($to, $subject, $message, $headers);
}

function install_to_cpanel($domain, $crt, $key, $cabundle)
{
    // Placeholder function for UAPI installation
    // In a real environment, this would use curl to talk to cPanel API
    return true;
}

// --- UI RENDERING ---

function render_head()
{
    global $tr, $lang;
    ?>
    <!DOCTYPE html>
    <html lang="es">

    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <meta name="sa-verify" content="sslfreexyz1wiae" />
        <title><?php echo $tr['title']; ?></title>
        <link rel="preconnect" href="https://fonts.googleapis.com">
        <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
        <link href="https://fonts.googleapis.com/css2?family=Outfit:wght@300;400;600&display=swap" rel="stylesheet">
        <style>
            :root {
                --primary: #6366f1;
                --primary-hover: #4f46e5;
                --bg: #0f172a;
                --card: #1e293b;
                --text: #f8fafc;
                --text-muted: #94a3b8;
                --success: #10b981;
                --error: #ef4444;
            }

            body {
                font-family: 'Outfit', sans-serif;
                background-color: var(--bg);
                color: var(--text);
                margin: 0;
                display: flex;
                align-items: center;
                justify-content: center;
                min-height: 100vh;
            }

            .container {
                width: 100%;
                max-width: 800px;
                padding: 2rem;
            }

            .card {
                background-color: var(--card);
                padding: 2.5rem;
                border-radius: 1.5rem;
                box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.5);
                border: 1px solid rgba(255, 255, 255, 0.1);
            }

            h1 {
                font-weight: 600;
                margin-bottom: 0.5rem;
                text-align: center;
                font-size: 2rem;
            }

            p.subtitle {
                color: var(--text-muted);
                text-align: center;
                margin-bottom: 2rem;
            }

            .form-group {
                margin-bottom: 1.5rem;
            }

            label {
                display: block;
                margin-bottom: 0.5rem;
                font-size: 0.875rem;
                color: var(--text-muted);
            }

            input[type="text"],
            input[type="password"],
            textarea {
                width: 100%;
                padding: 0.75rem 1rem;
                background: #0f172a;
                border: 1px solid rgba(255, 255, 255, 0.1);
                border-radius: 0.75rem;
                color: white;
                font-family: inherit;
                box-sizing: border-box;
            }

            textarea {
                height: 120px;
                font-family: monospace;
                font-size: 0.75rem;
                resize: vertical;
            }

            button {
                width: 100%;
                padding: 0.75rem;
                background-color: var(--primary);
                color: white;
                border: none;
                border-radius: 0.75rem;
                font-weight: 600;
                cursor: pointer;
                transition: all 0.2s;
                font-family: inherit;
            }

            button:hover {
                background-color: var(--primary-hover);
                transform: translateY(-1px);
            }

            .alert {
                padding: 1rem;
                border-radius: 0.75rem;
                margin-bottom: 1.5rem;
                font-size: 0.875rem;
            }

            .alert-success {
                background: rgba(16, 185, 129, 0.1);
                color: var(--success);
                border: 1px solid var(--success);
            }

            .alert-error {
                background: rgba(239, 68, 68, 0.1);
                color: var(--error);
                border: 1px solid var(--error);
            }

            .badge {
                display: inline-block;
                padding: 0.25rem 0.5rem;
                border-radius: 9999px;
                font-size: 0.75rem;
                font-weight: 600;
                text-transform: uppercase;
            }

            .badge-staging {
                background: #f59e0b;
                color: #78350f;
            }

            .badge-prod {
                background: #10b981;
                color: #064e3b;
            }

            .results {
                margin-top: 2rem;
                display: grid;
                gap: 1.5rem;
            }

            .copy-btn {
                margin-top: 0.5rem;
                background: rgba(255, 255, 255, 0.05);
                color: var(--text-muted);
                font-size: 0.75rem;
                padding: 0.4rem;
                width: auto;
                display: inline-block;
                border: 1px solid rgba(255, 255, 255, 0.1);
            }

            .logout {
                text-align: center;
                margin-top: 2rem;
            }

            .logout a {
                color: var(--text-muted);
                text-decoration: none;
                font-size: 0.875rem;
            }
        </style>
        <script>
            function copyToClipboard(id) {
                const el = document.getElementById(id);
                el.select();
                document.execCommand('copy');
                alert('<?php echo $tr['copy_success']; ?>');
            }

            function addCPanelSubdomains() {
                const input = document.querySelector('input[name="domain"]');
                let domain = input.value.trim();

                if (!domain) {
                    alert('<?php echo ($lang === 'es' ? 'Por favor ingresa el dominio principal primero (ej: midominio.com)' : 'Please enter the primary domain first (e.g., mydomain.com)'); ?>');
                    return;
                }

                // Get first domain if already a list
                const mainDomain = domain.split(',')[0].trim();
                const subs = ['autodiscover', 'mail', 'www'];
                let newList = [mainDomain];

                subs.forEach(sub => {
                    newList.push(sub + '.' + mainDomain);
                });

                input.value = newList.join(', ');
            }
        </script>
        <!-- Google tag (gtag.js) -->
        <script async src="https://www.googletagmanager.com/gtag/js?id=G-PYF2QF57XP"></script>
        <script>
            window.dataLayer = window.dataLayer || [];
            function gtag() { dataLayer.push(arguments); }
            gtag('js', new Date());
            gtag('config', 'G-PYF2QF57XP');
        </script>
    </head>

    <body>
        <?php
}

function render_login($error, $success = '', $notice = '')
{
    global $tr, $lang;
    render_head();
    ?>
        <div class="container" style="max-width: 400px;">
            <div style="text-align: right; margin-bottom: 1rem;">
                <a href="?lang=<?php echo $lang === 'es' ? 'en' : 'es'; ?>"
                    style="color: var(--primary); text-decoration: none; font-size: 0.875rem; font-weight: 600;">
                    <?php echo $tr['lang_toggle']; ?>
                </a>
            </div>
            <div class="card">
                <h1><?php echo $tr['login_title']; ?></h1>
                <p class="subtitle"><?php echo $tr['login_subtitle']; ?></p>
                <?php if ($error): ?>
                    <div class="alert alert-error">
                        <?php echo $error; ?>
                    </div>
                <?php endif; ?>
                <?php if ($success): ?>
                    <div class="alert alert-success">
                        <?php echo $success; ?>
                    </div>
                <?php endif; ?>
                <?php if ($notice): ?>
                    <div class="alert alert-error"
                        style="background: rgba(245, 158, 11, 0.1); color: #f59e0b; border-color: #f59e0b;">
                        <?php echo $notice; ?>
                    </div>
                <?php endif; ?>

                <?php if (isset($_GET['forgot'])): ?>
                    <p style="font-size: 0.875rem; color: var(--text-muted); margin-bottom: 1.5rem; text-align: center;">
                        <?php echo $tr['recovery_notice']; ?>
                    </p>
                    <form method="POST">
                        <button type="submit" name="recover_password"><?php echo $tr['recovery_btn']; ?></button>
                    </form>
                    <div style="text-align: center; margin-top: 1rem;">
                        <a href="index.php"
                            style="font-size: 0.875rem; color: var(--primary); text-decoration: none;"><?php echo $tr['back_to_login']; ?></a>
                    </div>
                <?php else: ?>
                    <form method="POST">
                        <div class="form-group">
                            <label><?php echo $tr['username']; ?></label>
                            <input type="text" name="username" required autofocus>
                        </div>
                        <div class="form-group">
                            <label><?php echo $tr['password']; ?></label>
                            <input type="password" name="password" required>
                        </div>
                        <button type="submit" name="login"><?php echo $tr['login_btn']; ?></button>
                    </form>
                    <div style="text-align: center; margin-top: 1.5rem;">
                        <a href="?forgot=1"
                            style="font-size: 0.875rem; color: var(--text-muted); text-decoration: none;"><?php echo $tr['forgot_pass']; ?></a>
                    </div>
                <?php endif; ?>

                <div
                    style="margin-top: 2rem; border-top: 1px solid rgba(255,255,255,0.1); padding-top: 1.5rem; text-align: center;">
                    <p style="font-size: 0.8rem; color: var(--text-muted); margin-bottom: 1rem;">
                            <?php echo $tr['like_tool']; ?>
                    </p>
                    <script type="text/javascript" src="https://cdnjs.buymeacoffee.com/1.0.0/button.prod.min.js"
                        data-name="bmc-button" data-slug="copypaste.com.mx" data-color="#FFDD00" data-emoji=""
                        data-font="Cookie" data-text="Buy me a coffee" data-outline-color="#000000"
                        data-font-color="#000000" data-coffee-color="#ffffff"></script>
                </div>

                <div style="margin-top: 2rem; text-align: center; font-size: 0.8rem; color: var(--text-muted);">
                    <?php echo $tr['created_by']; ?> <a href="https://mrcomps.com" target="_blank"
                        style="color: var(--primary); text-decoration: none; font-weight: 600;">Benito Mata</a>
                </div>
            </div>
        </div>
        <!-- 100% privacy-first analytics -->
        <script async src="https://scripts.simpleanalyticscdn.com/latest.js"></script>
    </body>

    </html>
    <?php
}

function render_mandatory_setup($error)
{
    global $tr, $lang;
    render_head();
    ?>
    <div class="container" style="max-width: 500px;">
        <div class="card">
            <h1><?php echo $tr['setup_title']; ?></h1>
            <p class="subtitle"><?php echo $tr['setup_subtitle']; ?></p>

            <?php if ($error): ?>
                <div class="alert alert-error">
                    <?php echo $error; ?>
                </div>
            <?php endif; ?>

            <form method="POST">
                <div class="form-group">
                    <label><?php echo $tr['setup_email_label']; ?></label>
                    <input type="text" name="setup_email" required placeholder="admin@tudominio.com">
                </div>
                <div class="form-group">
                    <label><?php echo $tr['setup_pass_label']; ?></label>
                    <input type="password" name="setup_password" required placeholder="••••••••">
                </div>
                <p style="font-size: 0.8rem; color: var(--text-muted); margin-bottom: 1.5rem;">
                    <?php echo $tr['setup_notice']; ?>
                </p>
                <button type="submit" name="mandatory_setup"><?php echo $tr['setup_btn']; ?></button>
            </form>
        </div>
    </div>
    </body>

    </html>
    <?php
}

function render_dashboard($status, $status_error, $certs)
{
    global $app_settings, $tr, $lang;
    render_head();
    ?>
    <div class="container">
        <div style="text-align: right; margin-bottom: 1rem;">
            <a href="?lang=<?php echo $lang === 'es' ? 'en' : 'es'; ?>"
                style="color: var(--primary); text-decoration: none; font-size: 0.875rem; font-weight: 600;">
                <?php echo $tr['lang_toggle']; ?>
            </a>
        </div>
        <div class="card">
            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 1rem;">
                <h1><?php echo $tr['title']; ?></h1>
                <span class="badge <?php echo LE_PRODUCTION ? 'badge-prod' : 'badge-staging'; ?>">
                    <?php echo LE_PRODUCTION ? $tr['production'] : $tr['staging']; ?>
                </span>
            </div>
            <p class="subtitle"><?php echo $tr['subtitle']; ?></p>

            <?php if ($status): ?>
                <div class="alert alert-success">
                    <?php echo $status; ?>
                </div>
            <?php endif; ?>
            <?php if ($status_error): ?>
                <div class="alert alert-error">
                    <?php echo $status_error; ?>
                </div>
            <?php endif; ?>
            <form method="POST">
                <div class="form-group">
                    <label style="display: flex; justify-content: space-between; align-items: center;">
                        <span><?php echo $tr['domains_label']; ?></span>
                        <a href="javascript:void(0)" onclick="addCPanelSubdomains()"
                            style="font-size: 0.75rem; color: var(--primary); text-decoration: none;"><?php echo $tr['add_subdomains']; ?></a>
                    </label>
                    <input type="text" name="domain" placeholder="<?php echo $tr['placeholder_domain']; ?>"
                        value="<?php echo $_POST['domain'] ?? ''; ?>">
                </div>
                <button type="submit" name="generate_ssl"><?php echo $tr['generate_btn']; ?></button>
            </form>

            <?php if ($certs['crt']): ?>
                <div class="results">
                    <div class="form-group">
                        <label>Certificate (CRT)</label>
                        <textarea id="res-crt" readonly><?php echo $certs['crt']; ?></textarea>
                        <button class="copy-btn" onclick="copyToClipboard('res-crt')"><?php echo $tr['copy_crt']; ?></button>
                    </div>
                    <div class="form-group">
                        <label>Private Key (KEY)</label>
                        <textarea id="res-key" readonly><?php echo $certs['key']; ?></textarea>
                        <button class="copy-btn" onclick="copyToClipboard('res-key')"><?php echo $tr['copy_key']; ?></button>
                    </div>
                    <div class="form-group">
                        <label>CA Bundle (CABUNDLE)</label>
                        <textarea id="res-bundle" readonly><?php echo $certs['bundle']; ?></textarea>
                        <button class="copy-btn"
                            onclick="copyToClipboard('res-bundle')"><?php echo $tr['copy_bundle']; ?></button>
                    </div>
                </div>
            <?php endif; ?>

            <div style="margin-top: 3rem; border-top: 1px solid rgba(255,255,255,0.1); padding-top: 2rem;">
                <h2 style="font-size: 1.25rem; font-weight: 600; margin-bottom: 1.5rem;"><?php echo $tr['user_settings']; ?>
                </h2>
                <form method="POST">
                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem;">
                        <div class="form-group">
                            <label><?php echo $tr['new_user']; ?></label>
                            <input type="text" name="new_username"
                                placeholder="<?php echo htmlspecialchars($app_settings['username']); ?>">
                        </div>
                        <div class="form-group">
                            <label><?php echo $tr['new_pass']; ?></label>
                            <input type="password" name="new_password" placeholder="••••••••">
                        </div>
                    </div>
                    <div class="form-group">
                        <label><?php echo $tr['recovery_email']; ?></label>
                        <input type="text" name="recovery_email"
                            value="<?php echo htmlspecialchars($app_settings['recovery_email']); ?>"
                            placeholder="admin@tudominio.com">
                    </div>
                    <div class="form-group"
                        style="display: flex; align-items: center; gap: 0.75rem; background: rgba(255,255,255,0.03); padding: 1rem; border-radius: 0.75rem; border: 1px solid rgba(255,255,255,0.05);">
                        <input type="checkbox" name="tracking_consent" id="tracking_consent" <?php echo $app_settings['tracking_consent'] ? 'checked' : ''; ?> style="width: auto;">
                        <label for="tracking_consent"
                            style="margin-bottom: 0; cursor: pointer;"><?php echo $tr['tracking_consent']; ?></label>
                    </div>
                    <button type="submit" name="update_settings"
                        style="background-color: rgba(99, 102, 241, 0.2); color: var(--primary); border: 1px solid var(--primary);"><?php echo $tr['update_btn']; ?></button>
                </form>
            </div>

            <div class="logout">
                <?php if (isset($_SESSION['authenticated'])): ?>
                    <a href="?logout=1"><?php echo $tr['logout']; ?></a>
                <?php else: ?>
                    <a href="index.php"><?php echo $tr['admin_login']; ?></a>
                <?php endif; ?>
            </div>

            <div
                style="margin-top: 2rem; border-top: 1px solid rgba(255,255,255,0.1); padding-top: 1.5rem; text-align: center; display: flex; flex-direction: column; align-items: center; gap: 1rem;">
                <script type="text/javascript" src="https://cdnjs.buymeacoffee.com/1.0.0/button.prod.min.js"
                    data-name="bmc-button" data-slug="copypaste.com.mx" data-color="#FFDD00" data-emoji=""
                    data-font="Cookie" data-text="Buy me a coffee" data-outline-color="#000000" data-font-color="#000000"
                    data-coffee-color="#ffffff"></script>

                <div style="font-size: 0.8rem; color: var(--text-muted);">
                    <?php echo $tr['created_by']; ?> <a href="https://mrcompus.com" target="_blank"
                        style="color: var(--primary); text-decoration: none; font-weight: 600;">Benito Mata</a>
                </div>

                <div
                    style="margin-top: 2rem; border-top: 1px solid rgba(255,255,255,0.1); padding-top: 1.5rem; text-align: left; width: 100%;">
                    <h3 style="font-size: 1rem; color: var(--text); margin-bottom: 1rem;">
                            <?php echo $tr['instructions_title']; ?></h3>
                    <ul
                        style="font-size: 0.875rem; color: var(--text-muted); padding-left: 1.2rem; margin: 0; line-height: 1.6;">
                        <li><?php echo $tr['instr_1']; ?></li>
                        <li><?php echo $tr['instr_2']; ?></li>
                        <li><?php echo $tr['instr_3']; ?></li>
                        <li><?php echo $tr['instr_4']; ?></li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
    <!-- 100% privacy-first analytics -->
    <script async src="https://scripts.simpleanalyticscdn.com/latest.js"></script>
    </body>
    <?php
}

render_dashboard($status_message, $status_error ?? '', $certs);
